///////////////////////////////////////////////////////////////////////////////////////
//
// MonoScan
//
// 27/04/07		Hounddog	Initial implementation
//

using System;
using System.Collections;
using System.ComponentModel;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Data;
using System.Windows.Forms;
using System.Text;

namespace MonoScan
{
	public struct ZoneStatusBit
	{
		public ZoneStatusBit(int index, string description)
		{
			Index = index;
			Description = description;
		}

		public int Index;
		public string Description;
	}

	/// <summary>
	/// Summary description for ZoneStatusIndicator.
	/// </summary>
	public class ZoneStatusIndicator : ZoneIndicator
	{
		const int marginPer = 5;			// % of border height/width
		const int cornerRadiusPer = 20;		// % of bit min(height/width)
		const int bitHeightPer = 170;		// % of text height
		const int bitWidthPer = 60;			// % of draw area width

		/// <summary>
		/// Required designer variable.
		/// </summary>
		private System.ComponentModel.Container components = null;

		public ZoneStatusIndicator()
		{
			// This call is required by the Windows.Forms Form Designer.
			InitializeComponent();

			SetStyle(ControlStyles.UserPaint, true);
			SetStyle(ControlStyles.AllPaintingInWmPaint, true);
			SetStyle(ControlStyles.DoubleBuffer, true);

			bits = null;

			length = 8;

			value = 0;

			UpdatePaint();
		}

		public string Caption
		{
			get { return caption; }

			set
			{
				if (value != caption)
				{
					caption = value;

					Invalidate();
				}
			}
		}

		protected string caption;

		public ZoneStatusBit[] Bits
		{
			get { return bits; }

			set
			{
				for (int index = 0; index < value.Length; ++ index)
				{
					int bitIndex = value[index].Index;

					if (bitIndex < 0 || bitIndex > 31)
						throw new ArgumentException("Invalid bits array");
				}

				bits = value;

				Invalidate();
			}
		}

		ZoneStatusBit[] bits;

		public int Length
		{
			get { return length; }

			set 
			{
				if (value != length)
				{
					if (length < 0 || length > 32)
						throw new ArgumentException("Invalid length");

					length = value;

					UpdatePaint();
				}
			}
		}

		int length;

		public override int Value
		{
			get { return value; }

			set
			{
				if (value != this.value)
				{
					this.value = value;

					UpdatePaint();
				}
			}
		}

		protected int value;

		/// <summary>
		/// Clean up any resources being used.
		/// </summary>
		protected override void Dispose( bool disposing )
		{
			if( disposing )
			{
				if( components != null )
					components.Dispose();
			}
			base.Dispose( disposing );
		}

		#region Component Designer generated code
		/// <summary>
		/// Required method for Designer support - do not modify 
		/// the contents of this method with the code editor.
		/// </summary>
		private void InitializeComponent()
		{
			components = new System.ComponentModel.Container();
		}
		#endregion

		protected virtual void UpdatePaint()
		{
			StringBuilder stringBuilder = new StringBuilder(32);

			for (int mask = 1 << length - 1; mask != 0; mask >>= 1)
				stringBuilder.Append((value & mask) != 0 ? '1' : '0');
		
			valueText = stringBuilder.ToString();

			Invalidate();
		}

		protected override void OnPaint(PaintEventArgs pe)
		{
			pe.Graphics.SmoothingMode = SmoothingMode.AntiAlias;

			Rectangle clientRectangle = ClientRectangle;

			// 3D border & margines

			if (clientRectangle.Width < 2 || clientRectangle.Height < 2)
				return;

			Rectangle borderRectangle = new Rectangle(clientRectangle.Left + 1, clientRectangle.Top + 1, 
				clientRectangle.Width - 2, clientRectangle.Height - 2);

			using (Pen pen = new Pen(SystemColors.ControlLightLight))
			{
				pe.Graphics.DrawLine(pen, borderRectangle.Left, borderRectangle.Bottom, 
					borderRectangle.Left, borderRectangle.Top);

				pe.Graphics.DrawLine(pen, borderRectangle.Left, borderRectangle.Top,
					borderRectangle.Right, borderRectangle.Top);
			}

			using (Pen pen = new Pen(SystemColors.ControlDark))
			{
				pe.Graphics.DrawLine(pen, borderRectangle.Right, borderRectangle.Top,
					borderRectangle.Right, borderRectangle.Bottom);

				pe.Graphics.DrawLine(pen, borderRectangle.Right, borderRectangle.Bottom,
					borderRectangle.Left, borderRectangle.Bottom);
			}

			float marginHor = borderRectangle.Width * marginPer / 100;
			float marginVer = borderRectangle.Height * marginPer / 100;

			if (borderRectangle.Height  < 2 * marginVer || borderRectangle.Width < 2 * marginHor)
				return;

			RectangleF drawRectangle = new RectangleF(borderRectangle.Left + marginHor,
				borderRectangle.Top + marginVer, borderRectangle.Width - 2 * marginHor,
				borderRectangle.Height - 2 * marginVer);

			using (Brush brush = new SolidBrush(SystemColors.WindowText))
			{
				// Caption

				float lineHeight = pe.Graphics.MeasureString("X", Font).Height;

				if (drawRectangle.Height < lineHeight)
					return;

				StringFormat stringFormat = new StringFormat();
				stringFormat.Alignment = StringAlignment.Center;
				stringFormat.LineAlignment = StringAlignment.Center;
				stringFormat.FormatFlags = StringFormatFlags.NoWrap;

				RectangleF captionRectangle = new RectangleF(drawRectangle.Left, drawRectangle.Top,
					drawRectangle.Width, lineHeight);

				pe.Graphics.DrawString(caption, Font, brush, captionRectangle, stringFormat);

				// Bits

				int bitCount = bits != null ? bits.Length : 0;

				float bitInterval = (drawRectangle.Height - lineHeight - marginVer) / (bitCount + 1);
				float bitHeight = bitHeightPer * lineHeight / 100;
				float bitWidth = bitWidthPer * drawRectangle.Width / 100;

				if (drawRectangle.Height >= lineHeight + marginVer + (bitCount + 1) * bitHeight)
				{
					RectangleF bitRectangle = new RectangleF(
						drawRectangle.Left + (drawRectangle.Width - bitWidth) / 2,
						drawRectangle.Top + lineHeight + marginVer + (bitInterval - bitHeight) / 2, 
						bitWidth, 
						bitHeight);

					GraphicsPath bitGraphicsPath = new GraphicsPath();

					float cornerRadius = cornerRadiusPer * Math.Min(bitHeight, bitWidth) / 100;

					bitGraphicsPath.AddArc(bitRectangle.Left, bitRectangle.Top, 2 * cornerRadius, 2 * cornerRadius,
						180, 90);
					bitGraphicsPath.AddLine(bitRectangle.Left + cornerRadius, bitRectangle.Top, 
						bitRectangle.Right - cornerRadius, bitRectangle.Top);
					bitGraphicsPath.AddArc(bitRectangle.Right - 2 * cornerRadius, bitRectangle.Top, 2 * cornerRadius, 
						2 * cornerRadius, 270, 90);
					bitGraphicsPath.AddLine(bitRectangle.Right, bitRectangle.Top + cornerRadius, 
						bitRectangle.Right, bitRectangle.Bottom - cornerRadius);
					bitGraphicsPath.AddArc(bitRectangle.Right - 2 * cornerRadius, bitRectangle.Bottom - 2 * cornerRadius,
						2 * cornerRadius, 2 * cornerRadius, 0, 90);
					bitGraphicsPath.AddLine(bitRectangle.Right - cornerRadius, bitRectangle.Bottom, 
						bitRectangle.Left + cornerRadius, bitRectangle.Bottom);
					bitGraphicsPath.AddArc(bitRectangle.Left, bitRectangle.Bottom - 2 * cornerRadius, 2 * cornerRadius, 
						2 * cornerRadius, 90, 90);
					bitGraphicsPath.AddLine(bitRectangle.Left, bitRectangle.Bottom - cornerRadius, 
						bitRectangle.Left, bitRectangle.Top + cornerRadius);

					Matrix matrix = new Matrix();
					matrix.Translate(0, bitInterval);

					using (Pen pen = new Pen(SystemColors.ControlDark))
					{
						using (Brush offBrush = new SolidBrush(Color.White), onBrush = new SolidBrush(Color.Red))
						{
							for (int index = 0; index < bitCount; ++ index)
							{
								ZoneStatusBit zoneStatusBit = bits[index];							

								Brush bitBrush = (value & 1 << zoneStatusBit.Index) != 0 ? onBrush : offBrush;

								pe.Graphics.FillPath(bitBrush, bitGraphicsPath);
								pe.Graphics.DrawPath(pen, bitGraphicsPath);

								pe.Graphics.DrawString(zoneStatusBit.Description, Font, brush, bitRectangle, stringFormat);

								bitRectangle.Y += bitInterval;							
								bitGraphicsPath.Transform(matrix);
							}
						}
					}

					// Value

					using (Font valueTextFont = new Font(Font, Font.Style | FontStyle.Bold))
					{
						pe.Graphics.DrawString(valueText, valueTextFont, brush, bitRectangle, stringFormat);
					}
				}
				else if (drawRectangle.Height >= lineHeight + marginVer + lineHeight)
				{
					// Value					

					using (Font valueTextFont = new Font(Font, Font.Style | FontStyle.Bold))
					{
						RectangleF valueTextRectangle = new RectangleF(
							drawRectangle.Left + (drawRectangle.Width - bitWidth) / 2,
							(drawRectangle.Top + marginVer + drawRectangle.Bottom) / 2, 
							bitWidth, 
							lineHeight);

						pe.Graphics.DrawString(valueText, valueTextFont, brush, valueTextRectangle, stringFormat);
					}
				}
			}
		}

		protected override void OnResize(EventArgs e)
		{
			Invalidate();
		}

		string valueText;
	}
}
